<?php

namespace App\Services;

use DateTime;
use Saleh7\Zatca\{
    CertificateBuilder,
    ZatcaAPI,
    InvoiceType,
    AdditionalDocumentReference,
    TaxScheme,
    PartyTaxScheme,
    Address,
    LegalEntity,
    Delivery,
    Party,
    PaymentMeans,
    TaxCategory,
    AllowanceCharge,
    TaxSubTotal,
    TaxTotal,
    LegalMonetaryTotal,
    ClassifiedTaxCategory,
    Item,
    Price,
    InvoiceLine,
    GeneratorInvoice,
    Invoice,
    UnitCode,
    InvoiceSigner,
    Storage,
    Helpers\Certificate,
    Exceptions\CertificateBuilderException,
    Exceptions\ZatcaApiException
};
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use App\Models\CompanyData;
use Illuminate\Support\Str; 
/**
 * فئة لخدمات الفواتير الإلكترونية ZATCA.
 * تتضمن وظائف لإنشاء الشهادات، طلبها، وتوليد الفواتير وتوقيعها.
 */
class ZatcaService
{
    /**
     * متغيرات الإعدادات التي يتم تحميلها من ملف config.
     *
     * @var array
     */
    protected $config;

    /**
     * مسارات الملفات التي يتم إنشاؤها.
     *
     * @var string
     */
    protected $outputPath;
    
    public function __construct()
    {
        $companyData = CompanyData::latest('id')->first();



        Config::set('zatca.organization_identifier', $companyData->Organaziation_Tax_ID);
        Config::set('zatca.common_name', $companyData->Organaziation_Common_Name);
        Config::set('zatca.organization_name', $companyData->Organaziation_Name);
        Config::set('zatca.organizational_unit_name', $companyData->Organaziation_Department);
        Config::set('zatca.address', $companyData->Organaziation_Address);
        Config::set('zatca.business_category', $companyData->Organaziation_Category);
        Config::set('zatca.is_production', $companyData->ZATCA_Production);
        
        // تحميل الإعدادات من ملف zatca.php
        $this->config = Config::get('zatca');
        $this->outputPath = $this->config['paths']['output_dir'];

        // التأكد من وجود مجلد الحفظ
        if (!File::isDirectory($this->outputPath)) {
            File::makeDirectory($this->outputPath, 0755, true);
        }
    }

    /**
     * الخطوة 1: توليد طلب الشهادة (CSR) والمفتاح الخاص (Private Key).
     *
     * @return array يحتوي على مسارات الملفات التي تم حفظها.
     * @throws CertificateBuilderException
     */
    public function generateCertificateAndPrivateKey(): array
    {
        // مسارات حفظ الملفات
        $csrPath = "{$this->outputPath}/{$this->config['paths']['csr_file']}";
        $privateKeyPath = "{$this->outputPath}/{$this->config['paths']['private_key_file']}";

        
    
        // استخدام فئة CertificateBuilder مع البيانات الديناميكية من ملف config
        (new CertificateBuilder())
            ->setOrganizationIdentifier($this->config['organization_identifier'])
            ->setSerialNumber($this->config['solution_name'], $this->config['model'], 'SME' . rand(10000, 99999)) // توليد رقم تسلسلي عشوائي للاختبار
            ->setCommonName($this->config['common_name'])
            ->setCountryName($this->config['country_name'])
            ->setOrganizationName($this->config['organization_name'])
            ->setOrganizationalUnitName($this->config['organizational_unit_name'])
            ->setAddress($this->config['address'])
            ->setInvoiceType($this->config['invoice_type'])
            ->setProduction($this->config['is_production'])
            ->setBusinessCategory($this->config['business_category'])
            ->generateAndSave($csrPath, $privateKeyPath);

        return [
            'csr_path' => $csrPath,
            'private_key_path' => $privateKeyPath
        ];
    }

    /**
     * الخطوة 2: طلب شهادة التوافق من ZATCA باستخدام الـ OTP.
     *
     * @param string $otp رمز OTP الذي تم الحصول عليه من بوابة ZATCA.
     * @return array يحتوي على الشهادة والمفتاح السري.
     * @throws ZatcaApiException
     */
    public function requestComplianceCertificate(string $otp): array
    {
        
    try{    
        // مسار ملف طلب الشهادة (CSR)
        $csrPath = "{$this->outputPath}/{$this->config['paths']['csr_file']}";
        $zatcaClient = new ZatcaAPI($this->config['is_production'] ? 'production' : 'sandbox');


        // تحميل ملف CSR وطلب الشهادة
        $csr = $zatcaClient->loadCSRFromFile($csrPath);   
        $complianceResult = $zatcaClient->requestComplianceCertificate($csr, $otp);

        // حفظ بيانات الشهادة والمفتاح السري في ملف JSON
        $certificateDataPath = "{$this->outputPath}/{$this->config['paths']['certificate_data_file']}";

        $zatcaClient->saveToJson(
            $complianceResult->getCertificate(),
            $complianceResult->getSecret(),
            $complianceResult->getRequestId(),
            $certificateDataPath
        );

        return [
            'certificate' => $complianceResult->getCertificate(),
            'secret' => $complianceResult->getSecret(),
            'request_id' => $complianceResult->getRequestId(),
            'certificate_file_path' => $certificateDataPath
        ];
        
       }catch(\Exception $ex){


  return [
            'certificate' => '',
            'secret' => '',
            'request_id' => '',
            'certificate_file_path' => ''
        ];

         }catch(\Error $ex){


  return [
            'certificate' => '',
            'secret' => '',
            'request_id' => '',
            'certificate_file_path' => ''
        ];

         }

        
    }

    /**
     * الخطوة 3 و 4: توليد الفاتورة وتوقيعها.
     *
     * @param array $invoiceData بيانات الفاتورة الديناميكية (مثل العميل، المنتجات، المبالغ).
     * @return array يحتوي على مسار الفاتورة الموقعة.
     * @throws \Exception
     */
    public function generateAndSignInvoiceXml(array $invoiceData): array
    {
        // مسار ملفات الشهادة والمفتاح
        $privateKeyPath = "{$this->outputPath}/{$this->config['paths']['private_key_file']}";
        $certificateDataPath = "{$this->outputPath}/{$this->config['paths']['certificate_data_file']}";
        $unsignedInvoicePath = "{$this->outputPath}/{$this->config['paths']['unsigned_invoice_file']}";
        $signedInvoicePath = "{$this->outputPath}/{$this->config['paths']['signed_invoice_file']}";

        // الخطوة 3: توليد الفاتورة XML غير الموقعة
        // هنا يتم استخدام البيانات الديناميكية المرسلة في $invoiceData
        $invoiceType = (new InvoiceType())
            ->setInvoice('standard')
            ->setInvoiceType('invoice');

        $taxScheme = (new TaxScheme())->setId("VAT");
        
        // بيانات المورد
        $supplierCompany = (new Party())
            ->setPartyIdentification($this->config['organization_identifier'])
            ->setPartyIdentificationId("CRN")
            ->setLegalEntity((new LegalEntity())->setRegistrationName($this->config['organization_name']))
            ->setPartyTaxScheme((new PartyTaxScheme())->setTaxScheme($taxScheme)->setCompanyId($this->config['organization_identifier']))
            ->setPostalAddress((new Address())->setStreetName($this->config['address'])->setCountry($this->config['country_name']));

        // بيانات العميل (ديناميكية)
        $customerLegalEntity = (new LegalEntity())->setRegistrationName($invoiceData['customer']['name']);
        $customerPartyTaxScheme = (new PartyTaxScheme())->setTaxScheme($taxScheme)->setCompanyId($invoiceData['customer']['tax_id']);
        $customerAddress = (new Address())->setStreetName($invoiceData['customer']['address'])->setCountry($invoiceData['customer']['country']);
        
        $customerParty = (new Party())
            ->setLegalEntity($customerLegalEntity)
            ->setPartyTaxScheme($customerPartyTaxScheme)
            ->setPostalAddress($customerAddress);

        // تفاصيل الفاتورة (ديناميكية)
        $invoiceLines = [];
        $totalTaxableAmount = 0;
        $totalTaxAmount = 0;

        foreach ($invoiceData['lines'] as $line) {
            $classifiedTax = (new ClassifiedTaxCategory())->setPercent($line['tax_rate'])->setTaxScheme($taxScheme);
            $productItem = (new Item())->setName($line['name'])->setClassifiedTaxCategory($classifiedTax);
            $price = (new Price())->setUnitCode(UnitCode::UNIT)->setPriceAmount($line['price']);
            $lineExtensionAmount = $line['price'] * $line['quantity'];
            $lineTaxAmount = $lineExtensionAmount * ($line['tax_rate'] / 100);

            $lineTaxTotal = (new TaxTotal())->setTaxAmount($lineTaxAmount)->setRoundingAmount($lineExtensionAmount + $lineTaxAmount);
            
            $invoiceLines[] = (new InvoiceLine())
                ->setId($line['id'])
                ->setItem($productItem)
                ->setLineExtensionAmount($lineExtensionAmount)
                ->setPrice($price)
                ->setTaxTotal($lineTaxTotal)
                ->setInvoicedQuantity($line['quantity']);

            $totalTaxableAmount += $lineExtensionAmount;
            $totalTaxAmount += $lineTaxAmount;
        }

        $taxSubTotal = (new TaxSubTotal)
            ->setTaxableAmount($totalTaxableAmount)
            ->setTaxAmount($totalTaxAmount)
            ->setTaxCategory((new TaxCategory())->setPercent(15)->setTaxScheme($taxScheme));

        $taxTotal = (new TaxTotal)
            ->addTaxSubTotal($taxSubTotal)
            ->setTaxAmount($totalTaxAmount);

        $legalMonetaryTotal = (new LegalMonetaryTotal())
            ->setLineExtensionAmount($totalTaxableAmount)
            ->setTaxExclusiveAmount($totalTaxableAmount)
            ->setTaxInclusiveAmount($totalTaxableAmount + $totalTaxAmount)
            ->setPayableAmount($totalTaxableAmount + $totalTaxAmount);

        
        $uuid = (string) Str::uuid();
        //$invoiceData['uuid']
        
        $invoice = (new Invoice())
            ->setUUID($uuid)
            ->setId($invoiceData['invoice_number'])
            ->setIssueDate(new DateTime($invoiceData['issue_date']))
            ->setIssueTime(new DateTime($invoiceData['issue_time']))
            ->setInvoiceType($invoiceType)
            ->setInvoiceCurrencyCode('SAR')
            ->setTaxCurrencyCode('SAR')
            ->setAccountingSupplierParty($supplierCompany)
            ->setAccountingCustomerParty($customerParty)
            ->setAdditionalDocumentReferences($this->createAdditionalDocumentReferences($invoiceData))
            ->setTaxTotal($taxTotal)
            ->setLegalMonetaryTotal($legalMonetaryTotal)
            ->setInvoiceLines($invoiceLines);

        $generatorXml = GeneratorInvoice::invoice($invoice);
        $unsignedXML = $generatorXml->getXML();

        (new Storage)->put($unsignedInvoicePath, $unsignedXML);

         
        
        
        // الخطوة 4: توقيع الفاتورة
        $jsonCertificate = (new Storage)->get($certificateDataPath);
        $jsonData = json_decode($jsonCertificate, true, 512, JSON_THROW_ON_ERROR);
        $certificate = $jsonData['certificate'];
        $secret = $jsonData['secret'];
        $privateKey = (new Storage)->get($privateKeyPath);

        $cleanPrivateKey = trim(str_replace(["-----BEGIN PRIVATE KEY-----", "-----END PRIVATE KEY-----"], "", $privateKey));

        $cert = new Certificate($certificate, $cleanPrivateKey, $secret);
        $signedInvoice = InvoiceSigner::signInvoice($unsignedXML, $cert)->getXML();

        (new Storage)->put($signedInvoicePath, $signedInvoice);

        
      
        
        $this->submitInvoiceToZatca($signedInvoicePath);
        
        //return ['signed_invoice_path' => $signedInvoicePath];
    }
    
    /**
     * الخطوة 5: إرسال الفاتورة الموقعة إلى ZATCA باستخدام cURL.
     *
     * @param string $signedInvoicePath مسار ملف الفاتورة الموقعة.
     * @return array
     * @throws \Exception
     */
    private function submitInvoiceToZatca(string $signedInvoicePath): array
    {
        // مسار ملف بيانات الشهادة
        $certificateDataPath = "{$this->outputPath}/{$this->config['paths']['certificate_data_file']}";
        
        // جلب بيانات الشهادة والمفتاح السري
        $jsonCertificate = (new Storage)->get($certificateDataPath);
        $jsonData = json_decode($jsonCertificate, true, 512, JSON_THROW_ON_ERROR);
        $certificate = $jsonData['certificate'];
        $secret = $jsonData['secret'];

        // جلب محتوى الفاتورة الموقعة
        $signedInvoiceXml = (new Storage)->get($signedInvoicePath);

        // إعداد بيانات الطلب
        $payload = [
            'invoice_base64' => base64_encode($signedInvoiceXml),
        ];

        // إعداد cURL
        $url = $this->config['is_production'] ? 'https://zatca.gov.sa/invoices/clearance' : 'https://zatca.gov.sa/sandbox/invoices/clearance';
        $ch = curl_init($url);
        
        // إعداد خيارات cURL
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept-Language: ar',
            'Accept: application/json',
            'Content-Type: application/json',
            'x-api-secret: ' . $secret,
            'x-api-token: ' . $certificate,
        ]);
        
        // تنفيذ الطلب
        $response = curl_exec($ch);
        
        // التحقق من وجود أخطاء في cURL
        if (curl_errno($ch)) {
            $error_msg = curl_error($ch);
            curl_close($ch);
            return [
                'status' => 'error',
                'message' => 'فشل في الاتصال بواجهة ZATCA API عبر cURL.',
                'details' => $error_msg
            ];
        }

        // إغلاق cURL
        curl_close($ch);
        
        // تحليل الاستجابة
        $responseData = json_decode($response, true);
        
        // التحقق من حالة الاستجابة
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        if ($httpCode === 200) {
            return [
                'status' => 'success',
                'data' => $responseData,
                'message' =>trans('admin.SendToZatcaSuccessfully')
            ];
            
                        
        } else {
            return [
                'status' => 'error',
                'message' => trans('admin.ErrorToSendToZatca'),
                'details' => $responseData
            ];
        }
    }
    
    /**
     * وظيفة مساعدة لإنشاء المراجع الإضافية.
     *
     * @param array $invoiceData
     * @return array
     */
    private function createAdditionalDocumentReferences(array $invoiceData): array
    {
        $additionalDocs = [];

        // المرجع الأول: رقم العداد
        $additionalDocs[] = (new AdditionalDocumentReference())->setId('ICV')->setUUID($invoiceData['invoice_counter_value']); // يتم جلب قيمة العداد من قاعدة البيانات.

        // المرجع الثاني: الـ Hash للفاتورة السابقة
        if (isset($invoiceData['previous_invoice_hash']) && !empty($invoiceData['previous_invoice_hash'])) {
            $additionalDocs[] = (new AdditionalDocumentReference())->setId('PIH')->setPreviousInvoiceHash($invoiceData['previous_invoice_hash']); // يتم جلب الـ Hash من الفاتورة السابقة في قاعدة البيانات.
        }

        // المرجع الثالث: الـ QR Code
        $additionalDocs[] = (new AdditionalDocumentReference())->setId('QR');
        
        return $additionalDocs;
    }
}
