<?php

namespace App\Listeners\Assets;

use App\Events\AssetsMonthlyEvent;
use App\Models\Journalizing;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use App\Traits\GeneralTrait;
use App\Models\JournalizingDetails;
use App\Models\GeneralDaily;
use App\Models\Assets;
use DB;
use DateTime;
use Illuminate\Support\Carbon;

class AssetsMonthlyListener
{


     use GeneralTrait;
    /**
     * Create the event listener.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * Handle the event.
     *
     * @param  \App\Events\TestEvent  $event
     * @return void
     */
    public function handle(AssetsMonthlyEvent $event)
    {



      // 1. Get all assets that need monthly depreciation.
        // We only need to check if the asset has already been depreciated this month.
        $currentMonth = Carbon::now()->format('Y-m');

        $itemsToDepreciate = Assets::where('Asset_Net', '>', 1)
            ->where('Asset_Type', 'مستهلك')
            ->where('Sale', 0)
            ->where('Delete', 0)
            ->where('Depreciation_Complex', '!=', null)
            ->where('Ehlak', '!=', null)
            // Add a condition to prevent running multiple times in the same month.
            // This assumes you have a column to track the last depreciation date, e.g., 'Last_Depreciation_Date'
            // For now, let's use a simplified check based on a 'Last_Depreciation_Month' column
            ->where('Last_Depreciation_Month', '!=', $currentMonth)
            ->get();
            
        // 2. Get the last Journalizing code once, before the loop.
        $lastJournal = Journalizing::orderByDesc('Code')->first();
        $codeT = $lastJournal ? $lastJournal->Code + 1 : 1;
        
        // 3. Loop through each asset and apply depreciation.
        foreach ($itemsToDepreciate as $item) {

            // Calculate the monthly depreciation amount.
            // If Annual_Depreciation is not numeric, default to 100/12.
            $annualDepreciation = is_numeric($item->Annual_Depreciation) && !empty($item->Annual_Depreciation)
                ? $item->Annual_Depreciation
                : 100;

            $monthlyDepreciation = $annualDepreciation / 12;

            // Start a database transaction to ensure data integrity.
            DB::beginTransaction();
            try {
                // 4. Create the main Journalizing entry.
                $journalEntryId = DB::table('journalizings')->insertGetId([
                    'Code'           => $codeT,
                    'Type'           => 'الهالك',
                    'TypeEn'         => 'Depreciation',
                    'Code_Type'      => $item->Code,
                    'Date'           => Carbon::now()->format('Y-m-d'),
                    'Draw'           => $item->Draw,
                    'Coin'           => $item->Coin,
                    'Cost_Center'    => $item->Cost_Center,
                    'Total_Debaitor' => $monthlyDepreciation,
                    'Total_Creditor' => $monthlyDepreciation,
                    'Note'           => $item->Note,
                ]);

                // 5. Create the two JournalizingDetails entries.
                // Debit entry
                JournalizingDetails::create([
                    'Joun_ID'   => $journalEntryId,
                    'Debitor'   => $monthlyDepreciation,
                    'Creditor'  => 0,
                    'Account'   => $item->Ehlak,
                    'Statement' => $item->Name,
                ]);

                // Credit entry
                JournalizingDetails::create([
                    'Joun_ID'   => $journalEntryId,
                    'Debitor'   => 0,
                    'Creditor'  => $monthlyDepreciation,
                    'Account'   => $item->Depreciation_Complex,
                    'Statement' => $item->Name,
                ]);

                // 6. Create the two GeneralDaily entries.
                // Debit entry
                GeneralDaily::create([
                    'Code'          => $codeT,
                    'Code_Type'     => $item->Code,
                    'Date'          => Carbon::now()->format('Y-m-d'),
                    'Type'          => 'الهالك',
                    'TypeEn'        => 'Depreciation',
                    'Debitor'       => $monthlyDepreciation,
                    'Creditor'      => 0,
                    'Statement'     => $item->Name,
                    'Draw'          => $item->Draw,
                    'Debitor_Coin'  => $item->Draw * $monthlyDepreciation,
                    'Creditor_Coin' => $item->Draw * 0,
                    'Account'       => $item->Ehlak,
                    'Coin'          => $item->Coin,
                    'Cost_Center'   => $item->Cost_Center,
                    'userr'         => auth()->guard('admin')->user()->id,
                ]);

                // Credit entry
                GeneralDaily::create([
                    'Code'          => $codeT,
                    'Code_Type'     => $item->Code,
                    'Date'          => Carbon::now()->format('Y-m-d'),
                    'Type'          => 'الهالك',
                    'TypeEn'        => 'Depreciation',
                    'Debitor'       => 0,
                    'Creditor'      => $monthlyDepreciation,
                    'Statement'     => $item->Name,
                    'Draw'          => $item->Draw,
                    'Debitor_Coin'  => $item->Draw * 0,
                    'Creditor_Coin' => $item->Draw * $monthlyDepreciation,
                    'Account'       => $item->Depreciation_Complex,
                    'Coin'          => $item->Coin,
                    'Cost_Center'   => $item->Cost_Center,
                    'userr'         => auth()->guard('admin')->user()->id,
                ]);

                // 7. Update the Asset's Net value and the last depreciation date.
                // We will use a dedicated column like 'Last_Depreciation_Month'
                $item->Asset_Net = $item->Asset_Net - $monthlyDepreciation;
                $item->Last_Depreciation_Month = $currentMonth;
                $item->save();

                // Increment the journal code for the next asset in the loop.
                $codeT++;

                // Commit the transaction if all queries succeed.
                DB::commit();

            } catch (\Exception $e) {
                DB::rollBack();
                // Log the error for the specific asset to help with debugging
                logger()->error("Failed to depreciate asset ID: {$item->id}. Error: " . $e->getMessage());
            }
        }


    }
}
